import type { APIRoute } from 'astro';
import { z } from 'zod';
import { sendEmail } from '../../utils/email';

const contactSchema = z.object({
  name: z.string().min(1, 'Name is required'),
  email: z.string().email('Invalid email address'),
  phone: z.string().optional(),
  message: z.string().min(1, 'Message is required'),
});

export const POST: APIRoute = async ({ request }) => {
  try {
    const data = await request.json();
    const validatedData = contactSchema.parse(data);
    
    await sendEmail(validatedData);

    return new Response(JSON.stringify({ success: true }), {
      status: 200,
      headers: { 'Content-Type': 'application/json' },
    });
  } catch (error) {
    console.error('Contact form error:', error);
    return new Response(
      JSON.stringify({ 
        success: false, 
        error: 'Failed to process contact form submission' 
      }), {
        status: 400,
        headers: { 'Content-Type': 'application/json' },
      }
    );
  }
};